const { getDb } = require('./init');

/**
 * Helper class to provide SQLite-like interface for MySQL2
 * This maintains backward compatibility while using MySQL2's promise-based API
 */
class MySQLHelper {
  constructor() {
    this.pool = null;
  }

  async getConnection() {
    if (!this.pool) {
      this.pool = getDb();
    }
    return await this.pool.getConnection();
  }

  /**
   * Execute a query that returns multiple rows (SELECT with multiple results)
   * @param {string} sql - SQL query
   * @param {Array} params - Query parameters
   * @param {Function} callback - Callback function (err, rows)
   */
  async all(sql, params = [], callback) {
    if (typeof params === 'function') {
      callback = params;
      params = [];
    }

    try {
      const connection = await this.getConnection();
      try {
        const sanitizedParams = this.sanitizeParams(params);
        const [rows] = await connection.execute(sql, sanitizedParams);
        callback(null, rows);
      } finally {
        connection.release();
      }
    } catch (error) {
      callback(error);
    }
  }

  /**
   * Execute a query that returns a single row (SELECT with single result)
   * @param {string} sql - SQL query
   * @param {Array} params - Query parameters
   * @param {Function} callback - Callback function (err, row)
   */
  async get(sql, params = [], callback) {
    if (typeof params === 'function') {
      callback = params;
      params = [];
    }

    try {
      const connection = await this.getConnection();
      try {
        const sanitizedParams = this.sanitizeParams(params);
        const [rows] = await connection.execute(sql, sanitizedParams);
        callback(null, rows[0] || null);
      } finally {
        connection.release();
      }
    } catch (error) {
      callback(error);
    }
  }

  /**
   * Sanitize parameters to convert undefined to null
   * @param {Array} params - Query parameters
   */
  sanitizeParams(params) {
    return params.map(param => param === undefined ? null : param);
  }

  /**
   * Execute a query that doesn't return rows (INSERT, UPDATE, DELETE)
   * @param {string} sql - SQL query
   * @param {Array} params - Query parameters
   * @param {Function} callback - Callback function (err, result)
   */
  async run(sql, params = [], callback) {
    if (typeof params === 'function') {
      callback = params;
      params = [];
    }

    try {
      const connection = await this.getConnection();
      try {
        // Sanitize parameters to convert undefined to null
        const sanitizedParams = this.sanitizeParams(params);
        const [result] = await connection.execute(sql, sanitizedParams);
        // Create SQLite-like result object
        const sqliteResult = {
          lastID: result.insertId,
          changes: result.affectedRows
        };
        callback(null, sqliteResult);
      } finally {
        connection.release();
      }
    } catch (error) {
      callback(error);
    }
  }

  /**
   * Begin a transaction
   */
  async beginTransaction() {
    const connection = await this.getConnection();
    await connection.beginTransaction();
    return connection;
  }

  /**
   * Commit a transaction
   */
  async commitTransaction(connection) {
    await connection.commit();
    connection.release();
  }

  /**
   * Rollback a transaction
   */
  async rollbackTransaction(connection) {
    await connection.rollback();
    connection.release();
  }
}

// Create and export a singleton instance
const mysqlHelper = new MySQLHelper();

module.exports = {
  MySQLHelper,
  mysqlHelper
};
